'use client';

import React, { useState, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import {
    Box,
    Container,
    Typography,
    Grid,
    Card,
    CardContent,
    Button,
    TextField,
    Divider,
    List,
    ListItem,
    ListItemText,
    Radio,
    RadioGroup,
    FormControlLabel,
    FormControl,
    FormLabel,
    Alert,
} from '@mui/material';
import { CreditCard, AccountBalance, CheckCircle } from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import { services } from '@/data/mockData';
import { serviceDetails } from '@/data/serviceDetails';

export const dynamic = 'force-dynamic';

function CheckoutContent() {
    const router = useRouter();
    const searchParams = useSearchParams();
    const serviceId = parseInt(searchParams.get('service') || '1');
    const packageIndex = parseInt(searchParams.get('package') || '1');

    const service = services.find((s) => s.id === serviceId);
    const details = serviceDetails[serviceId as keyof typeof serviceDetails];
    const selectedPackage = details?.packages?.[packageIndex];

    const [paymentMethod, setPaymentMethod] = useState('card');
    const [requirements, setRequirements] = useState('');
    const [errors, setErrors] = useState<{ requirements?: string }>({});

    if (!service || !selectedPackage) {
        return (
            <>
                <Header />
                <Container sx={{ py: 8, textAlign: 'center' }}>
                    <Typography variant="h4">Invalid order</Typography>
                    <Button variant="contained" onClick={() => router.push('/services')} sx={{ mt: 2 }}>
                        Browse Services
                    </Button>
                </Container>
                <Footer />
            </>
        );
    }

    const handlePlaceOrder = () => {
        if (!requirements.trim()) {
            setErrors({ requirements: 'Please provide project requirements' });
            return;
        }
        // Simulate order placement
        console.log('Order placed:', {
            service: service.id,
            package: selectedPackage.name,
            requirements,
            paymentMethod,
        });
        router.push('/order-confirmation?orderId=12345');
    };

    return (
        <>
            <Header />

            <Container maxWidth="lg" sx={{ mt: 4, mb: 8 }}>
                <Typography variant="h4" fontWeight={700} gutterBottom>
                    Checkout
                </Typography>
                <Typography variant="body1" color="text.secondary" sx={{ mb: 4 }}>
                    Complete your order
                </Typography>

                <Grid container spacing={4}>
                    {/* Left Column - Order Details */}
                    <Grid item xs={12} md={8}>
                        {/* Requirements */}
                        <Card sx={{ mb: 3 }}>
                            <CardContent>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Project Requirements
                                </Typography>
                                <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                                    Please provide detailed information about your project
                                </Typography>
                                <TextField
                                    fullWidth
                                    multiline
                                    rows={6}
                                    placeholder="Describe your project requirements, goals, and any specific details the freelancer should know..."
                                    value={requirements}
                                    onChange={(e) => {
                                        setRequirements(e.target.value);
                                        setErrors({});
                                    }}
                                    error={!!errors.requirements}
                                    helperText={errors.requirements}
                                />
                            </CardContent>
                        </Card>

                        {/* Payment Method */}
                        <Card>
                            <CardContent>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Payment Method
                                </Typography>
                                <FormControl component="fieldset" fullWidth>
                                    <RadioGroup
                                        value={paymentMethod}
                                        onChange={(e) => setPaymentMethod(e.target.value)}
                                    >
                                        <FormControlLabel
                                            value="card"
                                            control={<Radio />}
                                            label={
                                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                                    <CreditCard />
                                                    <span>Credit / Debit Card</span>
                                                </Box>
                                            }
                                        />
                                        <FormControlLabel
                                            value="paypal"
                                            control={<Radio />}
                                            label="PayPal"
                                        />
                                        <FormControlLabel
                                            value="bank"
                                            control={<Radio />}
                                            label={
                                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                                    <AccountBalance />
                                                    <span>Bank Transfer</span>
                                                </Box>
                                            }
                                        />
                                    </RadioGroup>
                                </FormControl>

                                {paymentMethod === 'card' && (
                                    <Box sx={{ mt: 3 }}>
                                        <Grid container spacing={2}>
                                            <Grid item xs={12}>
                                                <TextField fullWidth label="Card Number" placeholder="1234 5678 9012 3456" />
                                            </Grid>
                                            <Grid item xs={12} sm={6}>
                                                <TextField fullWidth label="Expiry Date" placeholder="MM/YY" />
                                            </Grid>
                                            <Grid item xs={12} sm={6}>
                                                <TextField fullWidth label="CVV" placeholder="123" />
                                            </Grid>
                                            <Grid item xs={12}>
                                                <TextField fullWidth label="Cardholder Name" />
                                            </Grid>
                                        </Grid>
                                    </Box>
                                )}
                            </CardContent>
                        </Card>
                    </Grid>

                    {/* Right Column - Order Summary */}
                    <Grid item xs={12} md={4}>
                        <Card sx={{ position: 'sticky', top: 20 }}>
                            <CardContent>
                                <Typography variant="h6" fontWeight={600} gutterBottom>
                                    Order Summary
                                </Typography>
                                <Divider sx={{ my: 2 }} />

                                <Box
                                    component="img"
                                    src={service.image}
                                    alt={service.title}
                                    sx={{
                                        width: '100%',
                                        height: 150,
                                        objectFit: 'cover',
                                        borderRadius: 1,
                                        mb: 2,
                                    }}
                                />

                                <Typography variant="subtitle1" fontWeight={600} gutterBottom>
                                    {service.title}
                                </Typography>
                                <Typography variant="body2" color="text.secondary" gutterBottom>
                                    by {service.provider.name}
                                </Typography>

                                <Divider sx={{ my: 2 }} />

                                <List dense>
                                    <ListItem sx={{ px: 0 }}>
                                        <ListItemText primary="Package" />
                                        <Typography fontWeight={600}>{selectedPackage.name}</Typography>
                                    </ListItem>
                                    <ListItem sx={{ px: 0 }}>
                                        <ListItemText primary="Delivery Time" />
                                        <Typography>{selectedPackage.deliveryTime}</Typography>
                                    </ListItem>
                                    <ListItem sx={{ px: 0 }}>
                                        <ListItemText primary="Revisions" />
                                        <Typography>{selectedPackage.revisions}</Typography>
                                    </ListItem>
                                </List>

                                <Divider sx={{ my: 2 }} />

                                <Box sx={{ mb: 2 }}>
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography>Subtotal</Typography>
                                        <Typography>${selectedPackage.price}</Typography>
                                    </Box>
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 1 }}>
                                        <Typography>Service Fee (10%)</Typography>
                                        <Typography>${(selectedPackage.price * 0.1).toFixed(2)}</Typography>
                                    </Box>
                                    <Divider sx={{ my: 1 }} />
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between' }}>
                                        <Typography variant="h6" fontWeight={700}>
                                            Total
                                        </Typography>
                                        <Typography variant="h6" fontWeight={700} color="primary.main">
                                            ${(selectedPackage.price * 1.1).toFixed(2)}
                                        </Typography>
                                    </Box>
                                </Box>

                                <Button
                                    variant="contained"
                                    size="large"
                                    fullWidth
                                    onClick={handlePlaceOrder}
                                    sx={{ mb: 2 }}
                                >
                                    Place Order
                                </Button>

                                <Alert severity="info" icon={<CheckCircle />}>
                                    Your payment is secured with escrow protection
                                </Alert>
                            </CardContent>
                        </Card>
                    </Grid>
                </Grid>
            </Container>

            <Footer />
        </>
    );
}

export default function CheckoutPage() {
    return (
        <Suspense fallback={<div>Loading...</div>}>
            <CheckoutContent />
        </Suspense>
    );
}
