'use client';

import React from 'react';
import { useRouter } from 'next/navigation';
import {
    Box,
    Container,
    Typography,
    Grid,
    Card,
    CardContent,
    Button,
    Avatar,
    Chip,
    Table,
    TableBody,
    TableCell,
    TableContainer,
    TableHead,
    TableRow,
    Paper,
    LinearProgress,
} from '@mui/material';
import {
    ShoppingCart,
    AttachMoney,
    Favorite,
    TrendingUp,
    Visibility,
    RateReview,
} from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import ServiceCard from '@/components/ServiceCard';
import { orders, services } from '@/data/mockData';

export default function DashboardPage() {
    const router = useRouter();
    const userOrders = orders.slice(0, 3);
    const recommendedServices = services.filter((s) => s.featured).slice(0, 3);

    const stats = [
        { label: 'Active Orders', value: '3', icon: <ShoppingCart />, color: '#6366f1' },
        { label: 'Total Spent', value: '$730', icon: <AttachMoney />, color: '#10b981' },
        { label: 'Saved Services', value: '12', icon: <Favorite />, color: '#ef4444' },
        { label: 'Reviews Given', value: '8', icon: <RateReview />, color: '#f59e0b' },
    ];

    const getStatusColor = (status) => {
        switch (status) {
            case 'in_progress':
                return 'primary';
            case 'delivered':
                return 'warning';
            case 'completed':
                return 'success';
            default:
                return 'default';
        }
    };

    const getStatusLabel = (status) => {
        return status.split('_').map((word) => word.charAt(0).toUpperCase() + word.slice(1)).join(' ');
    };

    return (
        <>
            <Header />

            <Box sx={{ bgcolor: 'background.default', minHeight: 'calc(100vh - 200px)' }}>
                <Container maxWidth="lg" sx={{ py: 4 }}>
                    {/* Welcome Section */}
                    <Box sx={{ mb: 4 }}>
                        <Typography variant="h4" fontWeight={700} gutterBottom>
                            Welcome back, John! 👋
                        </Typography>
                        <Typography variant="body1" color="text.secondary">
                            Here's what's happening with your orders today
                        </Typography>
                    </Box>

                    {/* Stats Cards */}
                    <Grid container spacing={3} sx={{ mb: 4 }}>
                        {stats.map((stat) => (
                            <Grid item xs={12} sm={6} md={3} key={stat.label}>
                                <Card>
                                    <CardContent>
                                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start' }}>
                                            <Box>
                                                <Typography variant="body2" color="text.secondary" gutterBottom>
                                                    {stat.label}
                                                </Typography>
                                                <Typography variant="h4" fontWeight={700}>
                                                    {stat.value}
                                                </Typography>
                                            </Box>
                                            <Box
                                                sx={{
                                                    width: 48,
                                                    height: 48,
                                                    borderRadius: 2,
                                                    bgcolor: `${stat.color}15`,
                                                    display: 'flex',
                                                    alignItems: 'center',
                                                    justifyContent: 'center',
                                                    color: stat.color,
                                                }}
                                            >
                                                {stat.icon}
                                            </Box>
                                        </Box>
                                    </CardContent>
                                </Card>
                            </Grid>
                        ))}
                    </Grid>

                    <Grid container spacing={3}>
                        {/* Recent Orders */}
                        <Grid item xs={12} md={8}>
                            <Card>
                                <CardContent>
                                    <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                                        <Typography variant="h6" fontWeight={600}>
                                            Recent Orders
                                        </Typography>
                                        <Button onClick={() => router.push('/dashboard/orders')}>View All</Button>
                                    </Box>

                                    <TableContainer>
                                        <Table>
                                            <TableHead>
                                                <TableRow>
                                                    <TableCell>Service</TableCell>
                                                    <TableCell>Freelancer</TableCell>
                                                    <TableCell>Status</TableCell>
                                                    <TableCell>Amount</TableCell>
                                                    <TableCell>Action</TableCell>
                                                </TableRow>
                                            </TableHead>
                                            <TableBody>
                                                {userOrders.map((order) => (
                                                    <TableRow key={order.id}>
                                                        <TableCell>
                                                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                                                                <Box
                                                                    component="img"
                                                                    src={order.thumbnail}
                                                                    alt={order.serviceName}
                                                                    sx={{ width: 50, height: 40, objectFit: 'cover', borderRadius: 1 }}
                                                                />
                                                                <Typography variant="body2" sx={{ maxWidth: 200 }}>
                                                                    {order.serviceName}
                                                                </Typography>
                                                            </Box>
                                                        </TableCell>
                                                        <TableCell>{order.freelancerName}</TableCell>
                                                        <TableCell>
                                                            <Chip
                                                                label={getStatusLabel(order.status)}
                                                                color={getStatusColor(order.status)}
                                                                size="small"
                                                            />
                                                        </TableCell>
                                                        <TableCell>${order.price}</TableCell>
                                                        <TableCell>
                                                            <Button size="small" startIcon={<Visibility />}>
                                                                View
                                                            </Button>
                                                        </TableCell>
                                                    </TableRow>
                                                ))}
                                            </TableBody>
                                        </Table>
                                    </TableContainer>
                                </CardContent>
                            </Card>
                        </Grid>

                        {/* Quick Actions */}
                        <Grid item xs={12} md={4}>
                            <Card sx={{ mb: 3 }}>
                                <CardContent>
                                    <Typography variant="h6" fontWeight={600} gutterBottom>
                                        Quick Actions
                                    </Typography>
                                    <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
                                        <Button
                                            variant="contained"
                                            fullWidth
                                            onClick={() => router.push('/services')}
                                        >
                                            Browse Services
                                        </Button>
                                        <Button
                                            variant="outlined"
                                            fullWidth
                                            onClick={() => router.push('/freelancers')}
                                        >
                                            Find Freelancers
                                        </Button>
                                        <Button
                                            variant="outlined"
                                            fullWidth
                                            onClick={() => router.push('/dashboard/favorites')}
                                        >
                                            My Favorites
                                        </Button>
                                        <Button
                                            variant="outlined"
                                            fullWidth
                                            onClick={() => router.push('/dashboard/profile')}
                                        >
                                            Edit Profile
                                        </Button>
                                    </Box>
                                </CardContent>
                            </Card>

                            <Card>
                                <CardContent>
                                    <Typography variant="h6" fontWeight={600} gutterBottom>
                                        Account Progress
                                    </Typography>
                                    <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                                        Complete your profile to get better recommendations
                                    </Typography>
                                    <LinearProgress variant="determinate" value={75} sx={{ mb: 1, height: 8, borderRadius: 4 }} />
                                    <Typography variant="body2" color="text.secondary">
                                        75% Complete
                                    </Typography>
                                </CardContent>
                            </Card>
                        </Grid>
                    </Grid>

                    {/* Recommended Services */}
                    <Box sx={{ mt: 4 }}>
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                            <Typography variant="h5" fontWeight={700}>
                                Recommended for You
                            </Typography>
                            <Button onClick={() => router.push('/services')}>View All</Button>
                        </Box>
                        <Grid container spacing={3}>
                            {recommendedServices.map((service) => (
                                <Grid item xs={12} sm={6} md={4} key={service.id}>
                                    <ServiceCard {...service} />
                                </Grid>
                            ))}
                        </Grid>
                    </Box>
                </Container>
            </Box>

            <Footer />
        </>
    );
}
