'use client';

import React, { useState } from 'react';
import {
    Box,
    Container,
    Typography,
    Accordion,
    AccordionSummary,
    AccordionDetails,
    Tabs,
    Tab,
    TextField,
    InputAdornment,
    Card,
    CardContent,
    Button,
} from '@mui/material';
import { ExpandMore, Search, HelpOutline } from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import { faqs } from '@/data/mockData';

export default function FAQPage() {
    const [selectedCategory, setSelectedCategory] = useState('General');
    const [searchQuery, setSearchQuery] = useState('');

    const categories = ['General', 'Payments', 'Freelancers', 'Clients'];

    const filteredFaqs = faqs.filter((faq) => {
        const matchesCategory = selectedCategory === 'General' ? true : faq.category === selectedCategory;
        const matchesSearch =
            faq.question.toLowerCase().includes(searchQuery.toLowerCase()) ||
            faq.answer.toLowerCase().includes(searchQuery.toLowerCase());
        return matchesCategory && matchesSearch;
    });

    return (
        <>
            <Header />

            {/* Page Header */}
            <Box
                sx={{
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    color: 'white',
                    py: 8,
                }}
            >
                <Container maxWidth="md" sx={{ textAlign: 'center' }}>
                    <HelpOutline sx={{ fontSize: 60, mb: 2 }} />
                    <Typography variant="h3" fontWeight={700} gutterBottom>
                        Frequently Asked Questions
                    </Typography>
                    <Typography variant="h6" sx={{ opacity: 0.9, mb: 4 }}>
                        Find answers to common questions about our platform
                    </Typography>

                    <TextField
                        fullWidth
                        placeholder="Search for answers..."
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                        InputProps={{
                            startAdornment: (
                                <InputAdornment position="start">
                                    <Search sx={{ color: 'white' }} />
                                </InputAdornment>
                            ),
                        }}
                        sx={{
                            bgcolor: 'rgba(255,255,255,0.15)',
                            borderRadius: 2,
                            '& .MuiOutlinedInput-root': {
                                color: 'white',
                                '& fieldset': { borderColor: 'rgba(255,255,255,0.3)' },
                                '&:hover fieldset': { borderColor: 'rgba(255,255,255,0.5)' },
                                '&.Mui-focused fieldset': { borderColor: 'white' },
                            },
                            '& .MuiInputLabel-root': { color: 'rgba(255,255,255,0.7)' },
                        }}
                    />
                </Container>
            </Box>

            <Container maxWidth="md" sx={{ mt: -4, mb: 8 }}>
                <Card>
                    <CardContent sx={{ p: 0 }}>
                        <Tabs
                            value={selectedCategory}
                            onChange={(e, v) => setSelectedCategory(v)}
                            variant="fullWidth"
                            sx={{ borderBottom: 1, borderColor: 'divider' }}
                        >
                            {categories.map((category) => (
                                <Tab key={category} label={category} value={category} />
                            ))}
                        </Tabs>

                        <Box sx={{ p: 3 }}>
                            {filteredFaqs.length === 0 ? (
                                <Box sx={{ textAlign: 'center', py: 6 }}>
                                    <Typography variant="h6" color="text.secondary">
                                        No questions found
                                    </Typography>
                                    <Typography variant="body2" color="text.secondary">
                                        Try adjusting your search or category filter
                                    </Typography>
                                </Box>
                            ) : (
                                filteredFaqs.map((faq) => (
                                    <Accordion key={faq.id}>
                                        <AccordionSummary expandIcon={<ExpandMore />}>
                                            <Typography fontWeight={600}>{faq.question}</Typography>
                                        </AccordionSummary>
                                        <AccordionDetails>
                                            <Typography color="text.secondary">{faq.answer}</Typography>
                                        </AccordionDetails>
                                    </Accordion>
                                ))
                            )}
                        </Box>
                    </CardContent>
                </Card>

                {/* Contact Support */}
                <Card sx={{ mt: 4, textAlign: 'center' }}>
                    <CardContent sx={{ p: 4 }}>
                        <Typography variant="h5" fontWeight={700} gutterBottom>
                            Still have questions?
                        </Typography>
                        <Typography variant="body1" color="text.secondary" sx={{ mb: 3 }}>
                            Can't find the answer you're looking for? Our support team is here to help.
                        </Typography>
                        <Button variant="contained" size="large" href="/contact">
                            Contact Support
                        </Button>
                    </CardContent>
                </Card>
            </Container>

            <Footer />
        </>
    );
}
