'use client';

import React, { useState } from 'react';
import {
    Box,
    Container,
    Typography,
    Grid,
    FormControl,
    InputLabel,
    Select,
    MenuItem,
    Slider,
    Checkbox,
    FormControlLabel,
    FormGroup,
    Button,
    TextField,
    InputAdornment,
    Chip,
    Paper,
} from '@mui/material';
import { Search, FilterList, Close } from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import FreelancerCard from '@/components/FreelancerCard';
import { freelancers, categories } from '@/data/mockData';

export default function FreelancersPage() {
    const [searchQuery, setSearchQuery] = useState('');
    const [selectedSkills, setSelectedSkills] = useState([]);
    const [hourlyRateRange, setHourlyRateRange] = useState([0, 200]);
    const [minRating, setMinRating] = useState(0);
    const [sortBy, setSortBy] = useState('rating');
    const [showPremiumOnly, setShowPremiumOnly] = useState(false);

    // Get all unique skills
    const allSkills = [...new Set(freelancers.flatMap((f) => f.skills))].sort();

    // Filter freelancers
    const filteredFreelancers = freelancers.filter((freelancer) => {
        const matchesSearch =
            freelancer.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
            freelancer.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
            freelancer.skills.some((skill) => skill.toLowerCase().includes(searchQuery.toLowerCase()));

        const matchesSkills =
            selectedSkills.length === 0 ||
            selectedSkills.some((skill) => freelancer.skills.includes(skill));

        const matchesRate =
            freelancer.hourlyRate >= hourlyRateRange[0] &&
            freelancer.hourlyRate <= hourlyRateRange[1];

        const matchesRating = freelancer.rating >= minRating;
        const matchesPremium = !showPremiumOnly || freelancer.isPremium;

        return matchesSearch && matchesSkills && matchesRate && matchesRating && matchesPremium;
    });

    // Sort freelancers
    const sortedFreelancers = [...filteredFreelancers].sort((a, b) => {
        if (sortBy === 'rating') return b.rating - a.rating;
        if (sortBy === 'rate_low') return a.hourlyRate - b.hourlyRate;
        if (sortBy === 'rate_high') return b.hourlyRate - a.hourlyRate;
        if (sortBy === 'projects') return b.completedProjects - a.completedProjects;
        return 0;
    });

    const handleSkillToggle = (skill) => {
        setSelectedSkills((prev) =>
            prev.includes(skill) ? prev.filter((s) => s !== skill) : [...prev, skill]
        );
    };

    const clearFilters = () => {
        setSearchQuery('');
        setSelectedSkills([]);
        setHourlyRateRange([0, 200]);
        setMinRating(0);
        setShowPremiumOnly(false);
        setSortBy('rating');
    };

    const activeFiltersCount =
        selectedSkills.length +
        (hourlyRateRange[0] !== 0 || hourlyRateRange[1] !== 200 ? 1 : 0) +
        (minRating > 0 ? 1 : 0) +
        (showPremiumOnly ? 1 : 0);

    return (
        <>
            <Header />

            {/* Page Header */}
            <Box
                sx={{
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    color: 'white',
                    py: 6,
                }}
            >
                <Container maxWidth="lg">
                    <Typography variant="h3" fontWeight={700} gutterBottom>
                        Find Freelancers
                    </Typography>
                    <Typography variant="h6" sx={{ opacity: 0.9 }}>
                        Browse {freelancers.length}+ talented professionals
                    </Typography>
                </Container>
            </Box>

            <Container maxWidth="lg" sx={{ mt: 4, mb: 8 }}>
                <Grid container spacing={3}>
                    {/* Filters Sidebar */}
                    <Grid item xs={12} md={3}>
                        <Paper sx={{ p: 3, position: 'sticky', top: 20 }}>
                            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                    <FilterList />
                                    <Typography variant="h6" fontWeight={600}>
                                        Filters
                                    </Typography>
                                    {activeFiltersCount > 0 && (
                                        <Chip label={activeFiltersCount} size="small" color="primary" />
                                    )}
                                </Box>
                                {activeFiltersCount > 0 && (
                                    <Button size="small" onClick={clearFilters} startIcon={<Close />}>
                                        Clear
                                    </Button>
                                )}
                            </Box>

                            {/* Search */}
                            <TextField
                                fullWidth
                                placeholder="Search freelancers..."
                                value={searchQuery}
                                onChange={(e) => setSearchQuery(e.target.value)}
                                InputProps={{
                                    startAdornment: (
                                        <InputAdornment position="start">
                                            <Search />
                                        </InputAdornment>
                                    ),
                                }}
                                sx={{ mb: 3 }}
                            />

                            {/* Premium Only */}
                            <FormControlLabel
                                control={
                                    <Checkbox
                                        checked={showPremiumOnly}
                                        onChange={(e) => setShowPremiumOnly(e.target.checked)}
                                    />
                                }
                                label="Premium Freelancers Only"
                                sx={{ mb: 2 }}
                            />

                            {/* Skills */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Skills
                            </Typography>
                            <FormGroup sx={{ mb: 3, maxHeight: 300, overflowY: 'auto' }}>
                                {allSkills.slice(0, 15).map((skill) => (
                                    <FormControlLabel
                                        key={skill}
                                        control={
                                            <Checkbox
                                                checked={selectedSkills.includes(skill)}
                                                onChange={() => handleSkillToggle(skill)}
                                            />
                                        }
                                        label={skill}
                                    />
                                ))}
                            </FormGroup>

                            {/* Hourly Rate Range */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Hourly Rate
                            </Typography>
                            <Box sx={{ px: 1, mb: 3 }}>
                                <Slider
                                    value={hourlyRateRange}
                                    onChange={(e, newValue) => setHourlyRateRange(newValue)}
                                    valueLabelDisplay="auto"
                                    min={0}
                                    max={200}
                                    valueLabelFormat={(value) => `$${value}`}
                                />
                                <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 1 }}>
                                    <Typography variant="caption">${hourlyRateRange[0]}/hr</Typography>
                                    <Typography variant="caption">${hourlyRateRange[1]}/hr</Typography>
                                </Box>
                            </Box>

                            {/* Rating */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Minimum Rating
                            </Typography>
                            <FormControl fullWidth>
                                <Select value={minRating} onChange={(e) => setMinRating(e.target.value)}>
                                    <MenuItem value={0}>All Ratings</MenuItem>
                                    <MenuItem value={4}>4+ Stars</MenuItem>
                                    <MenuItem value={4.5}>4.5+ Stars</MenuItem>
                                    <MenuItem value={4.8}>4.8+ Stars</MenuItem>
                                </Select>
                            </FormControl>
                        </Paper>
                    </Grid>

                    {/* Freelancers Grid */}
                    <Grid item xs={12} md={9}>
                        {/* Toolbar */}
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                            <Typography variant="body1" color="text.secondary">
                                {sortedFreelancers.length} freelancers found
                            </Typography>

                            <FormControl size="small" sx={{ minWidth: 180 }}>
                                <InputLabel>Sort By</InputLabel>
                                <Select value={sortBy} label="Sort By" onChange={(e) => setSortBy(e.target.value)}>
                                    <MenuItem value="rating">Highest Rated</MenuItem>
                                    <MenuItem value="projects">Most Projects</MenuItem>
                                    <MenuItem value="rate_low">Rate: Low to High</MenuItem>
                                    <MenuItem value="rate_high">Rate: High to Low</MenuItem>
                                </Select>
                            </FormControl>
                        </Box>

                        {/* Freelancers List */}
                        {sortedFreelancers.length === 0 ? (
                            <Paper sx={{ p: 6, textAlign: 'center' }}>
                                <Typography variant="h6" color="text.secondary" gutterBottom>
                                    No freelancers found
                                </Typography>
                                <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                                    Try adjusting your filters or search query
                                </Typography>
                                <Button variant="contained" onClick={clearFilters}>
                                    Clear All Filters
                                </Button>
                            </Paper>
                        ) : (
                            <Grid container spacing={3}>
                                {sortedFreelancers.map((freelancer) => (
                                    <Grid item xs={12} sm={6} md={4} key={freelancer.id}>
                                        <FreelancerCard {...freelancer} />
                                    </Grid>
                                ))}
                            </Grid>
                        )}

                        {/* Results Info */}
                        {sortedFreelancers.length > 0 && (
                            <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
                                <Typography variant="body2" color="text.secondary">
                                    Showing {sortedFreelancers.length} of {freelancers.length} freelancers
                                </Typography>
                            </Box>
                        )}
                    </Grid>
                </Grid>
            </Container>

            <Footer />
        </>
    );
}
