'use client';

import React, { useState } from 'react';
import { useRouter } from 'next/navigation';
import {
    Box,
    Container,
    Typography,
    TextField,
    Button,
    Card,
    CardContent,
    Checkbox,
    FormControlLabel,
    Divider,
    Link,
    InputAdornment,
    IconButton,
} from '@mui/material';
import { Visibility, VisibilityOff, Google, Facebook } from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';

export default function LoginPage() {
    const router = useRouter();
    const [showPassword, setShowPassword] = useState(false);
    const [formData, setFormData] = useState({
        email: '',
        password: '',
        remember: false,
    });
    const [errors, setErrors] = useState({});

    const handleChange = (e) => {
        const { name, value, checked } = e.target;
        setFormData((prev) => ({
            ...prev,
            [name]: name === 'remember' ? checked : value,
        }));
        // Clear error when user starts typing
        if (errors[name]) {
            setErrors((prev) => ({ ...prev, [name]: '' }));
        }
    };

    const validate = () => {
        const newErrors = {};
        if (!formData.email) {
            newErrors.email = 'Email is required';
        } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
            newErrors.email = 'Email is invalid';
        }
        if (!formData.password) {
            newErrors.password = 'Password is required';
        } else if (formData.password.length < 6) {
            newErrors.password = 'Password must be at least 6 characters';
        }
        return newErrors;
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        const newErrors = validate();
        if (Object.keys(newErrors).length > 0) {
            setErrors(newErrors);
        } else {
            // Simulate login - in real app, would call API
            console.log('Login:', formData);
            router.push('/dashboard');
        }
    };

    return (
        <>
            <Header />

            <Box
                sx={{
                    minHeight: 'calc(100vh - 200px)',
                    display: 'flex',
                    alignItems: 'center',
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    py: 8,
                }}
            >
                <Container maxWidth="sm">
                    <Card>
                        <CardContent sx={{ p: 4 }}>
                            <Typography variant="h4" fontWeight={700} gutterBottom textAlign="center">
                                Welcome Back
                            </Typography>
                            <Typography variant="body1" color="text.secondary" textAlign="center" sx={{ mb: 4 }}>
                                Log in to your account to continue
                            </Typography>

                            <form onSubmit={handleSubmit}>
                                <TextField
                                    fullWidth
                                    label="Email Address"
                                    name="email"
                                    type="email"
                                    value={formData.email}
                                    onChange={handleChange}
                                    error={!!errors.email}
                                    helperText={errors.email}
                                    sx={{ mb: 2 }}
                                />

                                <TextField
                                    fullWidth
                                    label="Password"
                                    name="password"
                                    type={showPassword ? 'text' : 'password'}
                                    value={formData.password}
                                    onChange={handleChange}
                                    error={!!errors.password}
                                    helperText={errors.password}
                                    InputProps={{
                                        endAdornment: (
                                            <InputAdornment position="end">
                                                <IconButton
                                                    onClick={() => setShowPassword(!showPassword)}
                                                    edge="end"
                                                >
                                                    {showPassword ? <VisibilityOff /> : <Visibility />}
                                                </IconButton>
                                            </InputAdornment>
                                        ),
                                    }}
                                    sx={{ mb: 2 }}
                                />

                                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                                    <FormControlLabel
                                        control={
                                            <Checkbox
                                                name="remember"
                                                checked={formData.remember}
                                                onChange={handleChange}
                                            />
                                        }
                                        label="Remember me"
                                    />
                                    <Link href="/forgot-password" underline="hover">
                                        Forgot password?
                                    </Link>
                                </Box>

                                <Button
                                    type="submit"
                                    variant="contained"
                                    size="large"
                                    fullWidth
                                    sx={{ mb: 2 }}
                                >
                                    Log In
                                </Button>

                                <Divider sx={{ my: 3 }}>OR</Divider>

                                <Button
                                    variant="outlined"
                                    size="large"
                                    fullWidth
                                    startIcon={<Google />}
                                    sx={{ mb: 2 }}
                                >
                                    Continue with Google
                                </Button>

                                <Button
                                    variant="outlined"
                                    size="large"
                                    fullWidth
                                    startIcon={<Facebook />}
                                >
                                    Continue with Facebook
                                </Button>

                                <Typography variant="body2" textAlign="center" sx={{ mt: 3 }}>
                                    Don't have an account?{' '}
                                    <Link href="/register" underline="hover" fontWeight={600}>
                                        Sign up
                                    </Link>
                                </Typography>
                            </form>
                        </CardContent>
                    </Card>
                </Container>
            </Box>

            <Footer />
        </>
    );
}
