'use client';

import React, { useState } from 'react';
import { useRouter } from 'next/navigation';
import {
    Box,
    Container,
    Typography,
    TextField,
    Button,
    Card,
    CardContent,
    Checkbox,
    FormControlLabel,
    Divider,
    Link,
    InputAdornment,
    IconButton,
    ToggleButton,
    ToggleButtonGroup,
    Alert,
} from '@mui/material';
import { Visibility, VisibilityOff, Google, Facebook, Person, Business } from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';

export default function RegisterPage() {
    const router = useRouter();
    const [showPassword, setShowPassword] = useState(false);
    const [userType, setUserType] = useState('client');
    const [formData, setFormData] = useState({
        name: '',
        email: '',
        password: '',
        confirmPassword: '',
        agreeTerms: false,
    });
    const [errors, setErrors] = useState({});

    const handleChange = (e) => {
        const { name, value, checked } = e.target;
        setFormData((prev) => ({
            ...prev,
            [name]: name === 'agreeTerms' ? checked : value,
        }));
        if (errors[name]) {
            setErrors((prev) => ({ ...prev, [name]: '' }));
        }
    };

    const validate = () => {
        const newErrors = {};
        if (!formData.name) newErrors.name = 'Name is required';
        if (!formData.email) {
            newErrors.email = 'Email is required';
        } else if (!/\S+@\S+\.\S+/.test(formData.email)) {
            newErrors.email = 'Email is invalid';
        }
        if (!formData.password) {
            newErrors.password = 'Password is required';
        } else if (formData.password.length < 6) {
            newErrors.password = 'Password must be at least 6 characters';
        }
        if (formData.password !== formData.confirmPassword) {
            newErrors.confirmPassword = 'Passwords do not match';
        }
        if (!formData.agreeTerms) {
            newErrors.agreeTerms = 'You must agree to the terms and conditions';
        }
        return newErrors;
    };

    const handleSubmit = (e) => {
        e.preventDefault();
        const newErrors = validate();
        if (Object.keys(newErrors).length > 0) {
            setErrors(newErrors);
        } else {
            console.log('Register:', { ...formData, userType });
            router.push(userType === 'freelancer' ? '/seller-dashboard' : '/dashboard');
        }
    };

    return (
        <>
            <Header />

            <Box
                sx={{
                    minHeight: 'calc(100vh - 200px)',
                    display: 'flex',
                    alignItems: 'center',
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    py: 8,
                }}
            >
                <Container maxWidth="sm">
                    <Card>
                        <CardContent sx={{ p: 4 }}>
                            <Typography variant="h4" fontWeight={700} gutterBottom textAlign="center">
                                Create Account
                            </Typography>
                            <Typography variant="body1" color="text.secondary" textAlign="center" sx={{ mb: 4 }}>
                                Join our marketplace today
                            </Typography>

                            <Box sx={{ mb: 3 }}>
                                <Typography variant="subtitle2" gutterBottom textAlign="center">
                                    I want to:
                                </Typography>
                                <ToggleButtonGroup
                                    value={userType}
                                    exclusive
                                    onChange={(e, value) => value && setUserType(value)}
                                    fullWidth
                                    sx={{ mb: 2 }}
                                >
                                    <ToggleButton value="client">
                                        <Person sx={{ mr: 1 }} />
                                        Hire Freelancers
                                    </ToggleButton>
                                    <ToggleButton value="freelancer">
                                        <Business sx={{ mr: 1 }} />
                                        Work as Freelancer
                                    </ToggleButton>
                                </ToggleButtonGroup>
                            </Box>

                            <form onSubmit={handleSubmit}>
                                <TextField
                                    fullWidth
                                    label="Full Name"
                                    name="name"
                                    value={formData.name}
                                    onChange={handleChange}
                                    error={!!errors.name}
                                    helperText={errors.name}
                                    sx={{ mb: 2 }}
                                />

                                <TextField
                                    fullWidth
                                    label="Email Address"
                                    name="email"
                                    type="email"
                                    value={formData.email}
                                    onChange={handleChange}
                                    error={!!errors.email}
                                    helperText={errors.email}
                                    sx={{ mb: 2 }}
                                />

                                <TextField
                                    fullWidth
                                    label="Password"
                                    name="password"
                                    type={showPassword ? 'text' : 'password'}
                                    value={formData.password}
                                    onChange={handleChange}
                                    error={!!errors.password}
                                    helperText={errors.password}
                                    InputProps={{
                                        endAdornment: (
                                            <InputAdornment position="end">
                                                <IconButton
                                                    onClick={() => setShowPassword(!showPassword)}
                                                    edge="end"
                                                >
                                                    {showPassword ? <VisibilityOff /> : <Visibility />}
                                                </IconButton>
                                            </InputAdornment>
                                        ),
                                    }}
                                    sx={{ mb: 2 }}
                                />

                                <TextField
                                    fullWidth
                                    label="Confirm Password"
                                    name="confirmPassword"
                                    type={showPassword ? 'text' : 'password'}
                                    value={formData.confirmPassword}
                                    onChange={handleChange}
                                    error={!!errors.confirmPassword}
                                    helperText={errors.confirmPassword}
                                    sx={{ mb: 2 }}
                                />

                                <FormControlLabel
                                    control={
                                        <Checkbox
                                            name="agreeTerms"
                                            checked={formData.agreeTerms}
                                            onChange={handleChange}
                                        />
                                    }
                                    label={
                                        <Typography variant="body2">
                                            I agree to the{' '}
                                            <Link href="/terms" underline="hover">
                                                Terms & Conditions
                                            </Link>{' '}
                                            and{' '}
                                            <Link href="/privacy" underline="hover">
                                                Privacy Policy
                                            </Link>
                                        </Typography>
                                    }
                                    sx={{ mb: 1 }}
                                />
                                {errors.agreeTerms && (
                                    <Alert severity="error" sx={{ mb: 2 }}>
                                        {errors.agreeTerms}
                                    </Alert>
                                )}

                                <Button
                                    type="submit"
                                    variant="contained"
                                    size="large"
                                    fullWidth
                                    sx={{ mb: 2 }}
                                >
                                    Create Account
                                </Button>

                                <Divider sx={{ my: 3 }}>OR</Divider>

                                <Button
                                    variant="outlined"
                                    size="large"
                                    fullWidth
                                    startIcon={<Google />}
                                    sx={{ mb: 2 }}
                                >
                                    Sign up with Google
                                </Button>

                                <Button
                                    variant="outlined"
                                    size="large"
                                    fullWidth
                                    startIcon={<Facebook />}
                                >
                                    Sign up with Facebook
                                </Button>

                                <Typography variant="body2" textAlign="center" sx={{ mt: 3 }}>
                                    Already have an account?{' '}
                                    <Link href="/login" underline="hover" fontWeight={600}>
                                        Log in
                                    </Link>
                                </Typography>
                            </form>
                        </CardContent>
                    </Card>
                </Container>
            </Box>

            <Footer />
        </>
    );
}
