'use client';

import React, { useState } from 'react';
import {
    Box,
    Container,
    Typography,
    Grid,
    FormControl,
    InputLabel,
    Select,
    MenuItem,
    Slider,
    Checkbox,
    FormControlLabel,
    FormGroup,
    Button,
    TextField,
    InputAdornment,
    Chip,
    Paper,
    ToggleButton,
    ToggleButtonGroup,
} from '@mui/material';
import {
    Search,
    GridView,
    ViewList,
    FilterList,
    Close,
} from '@mui/icons-material';
import Header from '@/components/Header';
import Footer from '@/components/Footer';
import ServiceCard from '@/components/ServiceCard';
import { services, categories } from '@/data/mockData';

export default function ServicesPage() {
    const [searchQuery, setSearchQuery] = useState('');
    const [selectedCategories, setSelectedCategories] = useState([]);
    const [priceRange, setPriceRange] = useState([0, 1000]);
    const [minRating, setMinRating] = useState(0);
    const [sortBy, setSortBy] = useState('popular');
    const [viewMode, setViewMode] = useState('grid');
    const [deliveryTime, setDeliveryTime] = useState('any');

    // Filter services
    const filteredServices = services.filter((service) => {
        const matchesSearch = service.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
            service.category.toLowerCase().includes(searchQuery.toLowerCase());
        const matchesCategory = selectedCategories.length === 0 ||
            selectedCategories.includes(service.category);
        const matchesPrice = service.price >= priceRange[0] && service.price <= priceRange[1];
        const matchesRating = service.rating >= minRating;

        let matchesDelivery = true;
        if (deliveryTime !== 'any') {
            const days = parseInt(service.deliveryTime);
            if (deliveryTime === '24h') matchesDelivery = days <= 1;
            else if (deliveryTime === '3days') matchesDelivery = days <= 3;
            else if (deliveryTime === '7days') matchesDelivery = days <= 7;
        }

        return matchesSearch && matchesCategory && matchesPrice && matchesRating && matchesDelivery;
    });

    // Sort services
    const sortedServices = [...filteredServices].sort((a, b) => {
        if (sortBy === 'price_low') return a.price - b.price;
        if (sortBy === 'price_high') return b.price - a.price;
        if (sortBy === 'rating') return b.rating - a.rating;
        if (sortBy === 'newest') return b.id - a.id;
        return 0; // popular (default order)
    });

    const handleCategoryToggle = (category) => {
        setSelectedCategories((prev) =>
            prev.includes(category)
                ? prev.filter((c) => c !== category)
                : [...prev, category]
        );
    };

    const clearFilters = () => {
        setSearchQuery('');
        setSelectedCategories([]);
        setPriceRange([0, 1000]);
        setMinRating(0);
        setDeliveryTime('any');
        setSortBy('popular');
    };

    const activeFiltersCount =
        selectedCategories.length +
        (priceRange[0] !== 0 || priceRange[1] !== 1000 ? 1 : 0) +
        (minRating > 0 ? 1 : 0) +
        (deliveryTime !== 'any' ? 1 : 0);

    return (
        <>
            <Header />

            {/* Page Header */}
            <Box
                sx={{
                    background: 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
                    color: 'white',
                    py: 6,
                }}
            >
                <Container maxWidth="lg">
                    <Typography variant="h3" fontWeight={700} gutterBottom>
                        Browse Services
                    </Typography>
                    <Typography variant="h6" sx={{ opacity: 0.9 }}>
                        Explore {services.length}+ professional services
                    </Typography>
                </Container>
            </Box>

            <Container maxWidth="lg" sx={{ mt: 4, mb: 8 }}>
                <Grid container spacing={3}>
                    {/* Filters Sidebar */}
                    <Grid item xs={12} md={3}>
                        <Paper sx={{ p: 3, position: 'sticky', top: 20 }}>
                            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                                    <FilterList />
                                    <Typography variant="h6" fontWeight={600}>
                                        Filters
                                    </Typography>
                                    {activeFiltersCount > 0 && (
                                        <Chip label={activeFiltersCount} size="small" color="primary" />
                                    )}
                                </Box>
                                {activeFiltersCount > 0 && (
                                    <Button size="small" onClick={clearFilters} startIcon={<Close />}>
                                        Clear
                                    </Button>
                                )}
                            </Box>

                            {/* Search */}
                            <TextField
                                fullWidth
                                placeholder="Search services..."
                                value={searchQuery}
                                onChange={(e) => setSearchQuery(e.target.value)}
                                InputProps={{
                                    startAdornment: (
                                        <InputAdornment position="start">
                                            <Search />
                                        </InputAdornment>
                                    ),
                                }}
                                sx={{ mb: 3 }}
                            />

                            {/* Categories */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Categories
                            </Typography>
                            <FormGroup sx={{ mb: 3 }}>
                                {categories.map((category) => (
                                    <FormControlLabel
                                        key={category.id}
                                        control={
                                            <Checkbox
                                                checked={selectedCategories.includes(category.name)}
                                                onChange={() => handleCategoryToggle(category.name)}
                                            />
                                        }
                                        label={
                                            <Box sx={{ display: 'flex', justifyContent: 'space-between', width: '100%' }}>
                                                <span>{category.name}</span>
                                                <span style={{ color: '#999', fontSize: '0.875rem' }}>({category.count})</span>
                                            </Box>
                                        }
                                    />
                                ))}
                            </FormGroup>

                            {/* Price Range */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Price Range
                            </Typography>
                            <Box sx={{ px: 1, mb: 3 }}>
                                <Slider
                                    value={priceRange}
                                    onChange={(e, newValue) => setPriceRange(newValue)}
                                    valueLabelDisplay="auto"
                                    min={0}
                                    max={1000}
                                    valueLabelFormat={(value) => `$${value}`}
                                />
                                <Box sx={{ display: 'flex', justifyContent: 'space-between', mt: 1 }}>
                                    <Typography variant="caption">${priceRange[0]}</Typography>
                                    <Typography variant="caption">${priceRange[1]}</Typography>
                                </Box>
                            </Box>

                            {/* Rating */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Minimum Rating
                            </Typography>
                            <FormControl fullWidth sx={{ mb: 3 }}>
                                <Select
                                    value={minRating}
                                    onChange={(e) => setMinRating(e.target.value)}
                                >
                                    <MenuItem value={0}>All Ratings</MenuItem>
                                    <MenuItem value={4}>4+ Stars</MenuItem>
                                    <MenuItem value={4.5}>4.5+ Stars</MenuItem>
                                    <MenuItem value={4.8}>4.8+ Stars</MenuItem>
                                </Select>
                            </FormControl>

                            {/* Delivery Time */}
                            <Typography variant="subtitle2" fontWeight={600} gutterBottom>
                                Delivery Time
                            </Typography>
                            <FormControl fullWidth>
                                <Select
                                    value={deliveryTime}
                                    onChange={(e) => setDeliveryTime(e.target.value)}
                                >
                                    <MenuItem value="any">Any Time</MenuItem>
                                    <MenuItem value="24h">24 Hours</MenuItem>
                                    <MenuItem value="3days">Up to 3 Days</MenuItem>
                                    <MenuItem value="7days">Up to 7 Days</MenuItem>
                                </Select>
                            </FormControl>
                        </Paper>
                    </Grid>

                    {/* Services Grid */}
                    <Grid item xs={12} md={9}>
                        {/* Toolbar */}
                        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
                            <Typography variant="body1" color="text.secondary">
                                {sortedServices.length} services found
                            </Typography>

                            <Box sx={{ display: 'flex', gap: 2, alignItems: 'center' }}>
                                <FormControl size="small" sx={{ minWidth: 150 }}>
                                    <InputLabel>Sort By</InputLabel>
                                    <Select
                                        value={sortBy}
                                        label="Sort By"
                                        onChange={(e) => setSortBy(e.target.value)}
                                    >
                                        <MenuItem value="popular">Most Popular</MenuItem>
                                        <MenuItem value="newest">Newest</MenuItem>
                                        <MenuItem value="rating">Highest Rated</MenuItem>
                                        <MenuItem value="price_low">Price: Low to High</MenuItem>
                                        <MenuItem value="price_high">Price: High to Low</MenuItem>
                                    </Select>
                                </FormControl>

                                <ToggleButtonGroup
                                    value={viewMode}
                                    exclusive
                                    onChange={(e, newMode) => newMode && setViewMode(newMode)}
                                    size="small"
                                >
                                    <ToggleButton value="grid">
                                        <GridView />
                                    </ToggleButton>
                                    <ToggleButton value="list">
                                        <ViewList />
                                    </ToggleButton>
                                </ToggleButtonGroup>
                            </Box>
                        </Box>

                        {/* Services List */}
                        {sortedServices.length === 0 ? (
                            <Paper sx={{ p: 6, textAlign: 'center' }}>
                                <Typography variant="h6" color="text.secondary" gutterBottom>
                                    No services found
                                </Typography>
                                <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                                    Try adjusting your filters or search query
                                </Typography>
                                <Button variant="contained" onClick={clearFilters}>
                                    Clear All Filters
                                </Button>
                            </Paper>
                        ) : (
                            <Grid container spacing={3}>
                                {sortedServices.map((service) => (
                                    <Grid item xs={12} sm={viewMode === 'grid' ? 6 : 12} lg={viewMode === 'grid' ? 4 : 12} key={service.id}>
                                        <ServiceCard {...service} />
                                    </Grid>
                                ))}
                            </Grid>
                        )}

                        {/* Pagination would go here */}
                        {sortedServices.length > 0 && (
                            <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
                                <Typography variant="body2" color="text.secondary">
                                    Showing {sortedServices.length} of {services.length} services
                                </Typography>
                            </Box>
                        )}
                    </Grid>
                </Grid>
            </Container>

            <Footer />
        </>
    );
}
