'use client';

import React from 'react';
import {
    Card,
    CardContent,
    Typography,
    Box,
    Avatar,
    Rating,
    Button,
    Chip,
} from '@mui/material';
import { LocationOn, WorkspacePremium } from '@mui/icons-material';
import Link from 'next/link';

interface FreelancerCardProps {
    id: number;
    name: string;
    title: string;
    avatar: string;
    rating: number;
    reviews: number;
    completedProjects: number;
    hourlyRate: number;
    skills: string[];
    location: string;
    isPremium?: boolean;
}

export default function FreelancerCard({
    id,
    name,
    title,
    avatar,
    rating,
    reviews,
    completedProjects,
    hourlyRate,
    skills,
    location,
    isPremium = false,
}: FreelancerCardProps) {
    return (
        <Card
            sx={{
                height: '100%',
                display: 'flex',
                flexDirection: 'column',
                position: 'relative',
            }}
        >
            <CardContent sx={{ textAlign: 'center', p: 3 }}>
                {isPremium && (
                    <Chip
                        icon={<WorkspacePremium />}
                        label="Premium"
                        size="small"
                        sx={{
                            position: 'absolute',
                            top: 12,
                            right: 12,
                            backgroundColor: 'secondary.main',
                            color: 'white',
                            fontWeight: 600,
                        }}
                    />
                )}

                <Avatar
                    src={avatar}
                    alt={name}
                    sx={{
                        width: 100,
                        height: 100,
                        margin: '0 auto 16px',
                        border: '4px solid',
                        borderColor: isPremium ? 'secondary.main' : 'primary.main',
                    }}
                />

                <Typography variant="h6" fontWeight={700} gutterBottom>
                    {name}
                </Typography>

                <Typography variant="body2" color="text.secondary" gutterBottom>
                    {title}
                </Typography>

                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'center', gap: 0.5, mb: 2 }}>
                    <Rating value={rating} precision={0.1} size="small" readOnly />
                    <Typography variant="caption" color="text.secondary">
                        {rating} ({reviews})
                    </Typography>
                </Box>

                <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'center', gap: 0.5, mb: 2 }}>
                    <LocationOn sx={{ fontSize: 16, color: 'text.secondary' }} />
                    <Typography variant="caption" color="text.secondary">
                        {location}
                    </Typography>
                </Box>

                <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 0.5, justifyContent: 'center', mb: 2 }}>
                    {skills.slice(0, 3).map((skill) => (
                        <Chip
                            key={skill}
                            label={skill}
                            size="small"
                            sx={{
                                backgroundColor: 'rgba(99, 102, 241, 0.1)',
                                color: 'primary.main',
                                fontSize: '0.7rem',
                            }}
                        />
                    ))}
                    {skills.length > 3 && (
                        <Chip
                            label={`+${skills.length - 3}`}
                            size="small"
                            sx={{
                                backgroundColor: 'rgba(99, 102, 241, 0.1)',
                                color: 'primary.main',
                                fontSize: '0.7rem',
                            }}
                        />
                    )}
                </Box>

                <Box
                    sx={{
                        display: 'flex',
                        justifyContent: 'space-around',
                        mb: 2,
                        py: 2,
                        borderTop: '1px solid',
                        borderBottom: '1px solid',
                        borderColor: 'divider',
                    }}
                >
                    <Box>
                        <Typography variant="h6" fontWeight={700} color="primary.main">
                            {completedProjects}
                        </Typography>
                        <Typography variant="caption" color="text.secondary">
                            Projects
                        </Typography>
                    </Box>
                    <Box>
                        <Typography variant="h6" fontWeight={700} color="primary.main">
                            ${hourlyRate}
                        </Typography>
                        <Typography variant="caption" color="text.secondary">
                            Per Hour
                        </Typography>
                    </Box>
                </Box>

                <Button
                    component={Link}
                    href={`/freelancers/${id}`}
                    variant="contained"
                    fullWidth
                >
                    View Profile
                </Button>
            </CardContent>
        </Card>
    );
}
